/*:
 * @target MZ
 * @plugindesc v1.0 保存方式を強制切替（auto/local/web）し、saveフォルダを自動作成＆詳細ログを出す診断プラグイン
 * @author HS
 *
 * @param Mode
 * @type select
 * @option auto @value auto
 * @option local (NW.jsに強制) @value local
 * @option web (localStorageに強制) @value web
 * @default auto
 *
 * @param EnsureSaveDirLocal
 * @text local時にsaveフォルダ自動作成
 * @type boolean
 * @default true
 *
 * @param VerboseLog
 * @text ログ詳細
 * @type boolean
 * @default true
 *
 * @help
 * ■ 目的
 * - 「セーブでブザーが鳴って保存されない」時に、保存先の問題かシリアライズ問題かを切り分けます。
 * - Mode=web にして保存できれば、ローカル書き込み（権限/フォルダ）が原因。
 * - Mode=local にして保存できれば、WebStorage（ブラウザ/容量）が原因。
 * - どちらでも失敗するなら、保存データのJSON化エラーや他プラグイン干渉の可能性が高いです。
 *
 * ■ 並び順
 * - できるだけ一番下（他のセーブ関連の「後ろ」）。
 */

(() => {
  'use strict';

  const PN = decodeURIComponent(document.currentScript.src.split('/').pop().replace(/\.js$/, ''));
  const P  = PluginManager.parameters(PN);
  const MODE = String(P.Mode || 'auto');
  const ENSURE_DIR = P.EnsureSaveDirLocal === 'true';
  const VLOG = P.VerboseLog === 'true';

  function vlog(...a){ if (VLOG) console.log('[HS_SaveModeSwitcher]', ...a); }
  function vwarn(...a){ console.warn('[HS_SaveModeSwitcher]', ...a); }
  function verr(...a){ console.error('[HS_SaveModeSwitcher]', ...a); }

  // ---- 1) 保存方式を強制
  const _isLocalMode = StorageManager.isLocalMode.bind(StorageManager);
  StorageManager.isLocalMode = function(){
    if (MODE === 'local') return true;
    if (MODE === 'web')   return false;
    return _isLocalMode(); // auto（通常の判定）
  };

  // ---- 2) local モード時に save フォルダを必ず作る
  function ensureSaveDir(){
    if (!StorageManager.isLocalMode()) return;
    try{
      // MZ公式のパスを取得
      const dir = StorageManager.fileDirectoryPath();
      const fs = require('fs');
      const path = require('path');
      if (!fs.existsSync(dir)){
        fs.mkdirSync(dir, { recursive:true });
        vlog('created save dir:', dir);
      }else{
        vlog('save dir exists:', dir);
      }
      // テスト書き込み
      const test = path.join(dir, '__hs_save_test.tmp');
      fs.writeFileSync(test, 'ok');
      fs.unlinkSync(test);
      vlog('write test ok at', dir);
    }catch(e){
      verr('ensureSaveDir failed (permission/path):', e);
    }
  }

  // 起動時に一度だけチェック
  const _bootStart = Scene_Boot.prototype.start;
  Scene_Boot.prototype.start = function(){
    _bootStart.call(this);
    vlog('mode=', MODE, 'localModeDetected=', StorageManager.isLocalMode());
    if (ENSURE_DIR) ensureSaveDir();

    // WebStorage時の容量テスト（軽く）
    if (!StorageManager.isLocalMode()){
      try{
        const key = '__hs_web_test';
        const big = 'x'.repeat(1024 * 100); // 100KB
        window.localStorage.setItem(key, big);
        window.localStorage.removeItem(key);
        vlog('webstorage test ok (~100KB)');
      }catch(e){
        vwarn('webstorage setItem failed:', e);
      }
    }
  };

  // ---- 3) 詳細ログ（成功/失敗の明示）
  const _saveGame = DataManager.saveGame;
  DataManager.saveGame = function(savefileId){
    vlog('saveGame called:', savefileId, 'backend=', StorageManager.isLocalMode()?'local':'web');
    try{
      const r = _saveGame.call(this, savefileId);
      if (r && typeof r.then === 'function'){
        return r.then(()=>{
          vlog('save success:', savefileId);
          return true;
        }).catch(e=>{
          verr('save FAILED:', e);
          return false;
        });
      }else{
        vlog('save returned:', r);
        return r;
      }
    }catch(e){
      verr('save threw:', e);
      return Promise.reject(e);
    }
  };

  const _loadGame = DataManager.loadGame;
  DataManager.loadGame = function(savefileId){
    vlog('loadGame called:', savefileId, 'backend=', StorageManager.isLocalMode()?'local':'web');
    try{
      const r = _loadGame.call(this, savefileId);
      if (r && typeof r.then === 'function'){
        return r.then(()=>{
          vlog('load success:', savefileId);
          return true;
        }).catch(e=>{
          verr('load FAILED:', e);
          return false;
        });
      }else{
        vlog('load returned:', r);
        return r;
      }
    }catch(e){
      verr('load threw:', e);
      return Promise.reject(e);
    }
  };

  // セーブ失敗時にブザーで終わらせず、理由を確実に出す
  const _execSave = Scene_File.prototype.executeSave;
  Scene_File.prototype.executeSave = function(id){
    try{ $gameSystem.onBeforeSave(); }catch(e){ vwarn('onBeforeSave error:', e); }
    const p = DataManager.saveGame(id);
    if (p && typeof p.then === 'function'){
      p.then(()=>this.onSaveSuccess()).catch(e=>{
        verr('onSaveFailure detail:', e);
        SoundManager.playBuzzer();
        this.onSaveFailure();
      });
    }else{
      if (p){ this.onSaveSuccess(); }
      else { vwarn('onSaveFailure (no promise, false returned)'); SoundManager.playBuzzer(); this.onSaveFailure(); }
    }
  };

})();
